//*************************************************************************************************
//
//	Description:
//		Basic billboarding shader
//		
//		
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Matt Hobbs
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		Matt             09/08/2006  Created
//		TMann			 08/02/2007	 1.2			Added GL/PS3
//
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_

//-----------------------------------------------------------------------
//
// Lighting
//
#if defined(_XBOX) && !defined(_TOOLS_COMPILATION_)
#include "D:\Render\Shaders\stddefs.fxh"
#include "D:\Render\Shaders\lighting_globals.fxh"
#else
#include "..\..\..\Render\Shaders\stddefs.fxh"
#include "..\..\..\Render\Shaders\lighting_globals.fxh"
#endif
DECLARE_LIGHTING_PARAMS

#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
#else
#define _MINFILTER	Linear
#define SET_NO_ANISOTROPY MaxAnisotropy = 1;
#endif



//-----------------------------------------------------------------------
//
// Transforms
//
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 projMatrix : Projection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;




//-----------------------------------------------------------------------
//
// Textures
//
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;




//-----------------------------------------------------------------------
//
// Misc params
//
float4 colourModulation
<
	bool appEdit = true;
> = float4(1.0f, 1.0f, 1.0f, 1.0f);




//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Clamp";
>
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Clamp;
#endif
};






//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

struct VSINPUT
{
	float3 position : POSITION;												
	float3 rotSize	: NORMAL;														// width, height and rotation (radians)
	float2 texCoord : TEXCOORD0;											
	float4 colour   : COLOR0;												

};

struct VSOUTPUT
{
	float4 position		: POSITION;												
	float2 texCoord		: TEXCOORD0;											
	float4 colour		: TEXCOORD1;												
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT VShader( VSINPUT _input )
{
	VSOUTPUT _output;
	
	float sn, cs;
	
	//get rotation params for 2D matrix mul
	sn = sin(_input.rotSize.z);
	cs = cos(_input.rotSize.z);
	
	// get rotated corner offset in clip space	
	float4 cornerOfs = 
		mul( float4( (cs * _input.rotSize.x) - (sn * _input.rotSize.y), (sn * _input.rotSize.x + cs * _input.rotSize.y), 0.0f, 0.0f),
			 projMatrix);

	// get centre position in clip space and combine with offset
	float4 inPos = float4( _input.position, 1.0f );
	_output.position = mul( inPos, worldviewproj );
	_output.position += cornerOfs;
	
	// apply lighting
	_output.colour = float4(1.0f, 1.0f, 1.0f, 1.0f);
	float3 normal = float3(0.0f, 1.0f, 0.0f);
	// note inPos == worldPos as world is identity
	DO_VERTEX_LIGHTING( inPos, normal, _output.colour )
	_output.colour *= _input.colour;

	// pass thru
	_output.texCoord = _input.texCoord;

	return _output;
}

VSOUTPUT VShaderUnlit( VSINPUT _input )
{
	VSOUTPUT _output;
	
	float sn, cs;
	
	//get rotation params for 2D matrix mul
	sn = sin(_input.rotSize.z);
	cs = cos(_input.rotSize.z);
	
	// get rotated corner offset in clip space	
	float4 cornerOfs = 
		mul( float4( (cs * _input.rotSize.x) - (sn * _input.rotSize.y), (sn * _input.rotSize.x + cs * _input.rotSize.y), 0.0f, 0.0f),
			 projMatrix);

	// get centre position in clip space and combine with offset
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.position += cornerOfs;

	// pass thru
	_output.colour = _input.colour * colourModulation;
	_output.texCoord = _input.texCoord;

	return _output;
}


//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

struct PSINPUT
{
	float2 texCoord					: TEXCOORD0;												
	float4 colour					: TEXCOORD1;													
	
};

struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT fragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	
	// modulate with interpolated vertex colour
	_output.colour  = diffuseTexColour * _input.colour;
	
	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Billboard
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Billboard";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx VShader();
		PixelShader = compile sce_fp_rsx fragmentShader();
#else
		VertexShader = compile vs_3_0 VShader();
		PixelShader = compile ps_3_0 fragmentShader();
#endif
	}
}

technique BillboardUnlit
<
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "BillboardUnlit";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx VShaderUnlit();
		PixelShader = compile sce_fp_rsx fragmentShader();
#else
		VertexShader = compile vs_3_0 VShaderUnlit();
		PixelShader = compile ps_3_0 fragmentShader();
#endif
	}
}
